/*
 * Decompiled with CFR 0.152.
 */
package org.openqa.selenium.grid.distributor;

import com.google.common.collect.ImmutableSet;
import java.time.Instant;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.logging.Logger;
import org.openqa.selenium.events.EventBus;
import org.openqa.selenium.grid.config.Config;
import org.openqa.selenium.grid.data.Availability;
import org.openqa.selenium.grid.data.NodeDrainStarted;
import org.openqa.selenium.grid.data.NodeId;
import org.openqa.selenium.grid.data.NodeRemovedEvent;
import org.openqa.selenium.grid.data.NodeRestartedEvent;
import org.openqa.selenium.grid.data.NodeStatus;
import org.openqa.selenium.grid.data.Session;
import org.openqa.selenium.grid.data.SessionClosedEvent;
import org.openqa.selenium.grid.data.Slot;
import org.openqa.selenium.grid.data.SlotId;
import org.openqa.selenium.grid.server.EventBusOptions;
import org.openqa.selenium.internal.Debug;
import org.openqa.selenium.internal.Require;
import org.openqa.selenium.remote.SessionId;

public class GridModel {
    private static final SessionId RESERVED = new SessionId("reserved");
    private static final Logger LOG = Logger.getLogger(GridModel.class.getName());
    private static final int PURGE_TIMEOUT_MULTIPLIER = 4;
    private static final int UNHEALTHY_THRESHOLD = 4;
    private final ReadWriteLock lock = new ReentrantReadWriteLock(true);
    private final Set<NodeStatus> nodes = Collections.newSetFromMap(new ConcurrentHashMap());
    private final Map<NodeId, Instant> nodePurgeTimes = new ConcurrentHashMap<NodeId, Instant>();
    private final Map<NodeId, Integer> nodeHealthCount = new ConcurrentHashMap<NodeId, Integer>();
    private final EventBus events;

    public GridModel(EventBus events) {
        this.events = Require.nonNull("Event bus", events);
        this.events.addListener(NodeDrainStarted.listener(nodeId -> this.setAvailability((NodeId)nodeId, Availability.DRAINING)));
        this.events.addListener(SessionClosedEvent.listener(this::release));
    }

    public static GridModel create(Config config) {
        EventBus bus = new EventBusOptions(config).getEventBus();
        return new GridModel(bus);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void add(NodeStatus node) {
        Require.nonNull("Node", node);
        Lock writeLock = this.lock.writeLock();
        writeLock.lock();
        try {
            Iterator<NodeStatus> iterator = this.nodes.iterator();
            while (iterator.hasNext()) {
                NodeStatus next = iterator.next();
                if (next.getNodeId().equals(node.getNodeId()) && next.getExternalUri().equals(node.getExternalUri())) {
                    iterator.remove();
                    LOG.log(Debug.getDebugLogLevel(), "Refreshing node with id {0}", node.getNodeId());
                    NodeStatus refreshed = this.rewrite(node, next.getAvailability());
                    this.nodes.add(refreshed);
                    this.nodePurgeTimes.put(refreshed.getNodeId(), Instant.now());
                    this.updateHealthCheckCount(refreshed.getNodeId(), refreshed.getAvailability());
                    return;
                }
                if (!next.getNodeId().equals(node.getNodeId()) && next.getExternalUri().equals(node.getExternalUri())) {
                    LOG.info(String.format("Re-adding node with id %s and URI %s.", node.getNodeId(), node.getExternalUri()));
                    this.events.fire(new NodeRestartedEvent(node));
                    iterator.remove();
                    break;
                }
                if (!next.getNodeId().equals(node.getNodeId())) continue;
                LOG.info(String.format("Re-adding node with id %s and URI %s.", node.getNodeId(), node.getExternalUri()));
                iterator.remove();
                break;
            }
            LOG.log(Debug.getDebugLogLevel(), "Adding node with id {0} and URI {1}", new Object[]{node.getNodeId(), node.getExternalUri()});
            NodeStatus refreshed = this.rewrite(node, Availability.DOWN);
            this.nodes.add(refreshed);
            this.nodePurgeTimes.put(refreshed.getNodeId(), Instant.now());
            this.updateHealthCheckCount(refreshed.getNodeId(), refreshed.getAvailability());
        }
        finally {
            writeLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void refresh(NodeStatus status) {
        Require.nonNull("Node status", status);
        Lock writeLock = this.lock.writeLock();
        writeLock.lock();
        try {
            Iterator<NodeStatus> iterator = this.nodes.iterator();
            while (iterator.hasNext()) {
                NodeStatus node = iterator.next();
                if (!node.getNodeId().equals(status.getNodeId())) continue;
                iterator.remove();
                if (node.getAvailability() == Availability.DOWN) {
                    this.nodes.add(this.rewrite(status, Availability.DOWN));
                } else {
                    this.nodes.add(status);
                }
                this.nodePurgeTimes.put(status.getNodeId(), Instant.now());
                return;
            }
        }
        finally {
            writeLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void touch(NodeStatus nodeStatus) {
        Require.nonNull("Node ID", nodeStatus);
        Lock writeLock = this.lock.writeLock();
        writeLock.lock();
        try {
            NodeStatus node = this.getNode(nodeStatus.getNodeId());
            if (node != null) {
                this.nodePurgeTimes.put(node.getNodeId(), Instant.now());
                if (node.getAvailability() != nodeStatus.getAvailability() && nodeStatus.getAvailability() == Availability.UP) {
                    this.nodes.remove(node);
                    this.nodes.add(nodeStatus);
                }
            }
        }
        finally {
            writeLock.unlock();
        }
    }

    public void remove(NodeId id) {
        Require.nonNull("Node ID", id);
        Lock writeLock = this.lock.writeLock();
        writeLock.lock();
        try {
            this.nodes.removeIf(n -> n.getNodeId().equals(id));
            this.nodePurgeTimes.remove(id);
            this.nodeHealthCount.remove(id);
        }
        finally {
            writeLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void purgeDeadNodes() {
        Lock writeLock = this.lock.writeLock();
        writeLock.lock();
        try {
            HashMap<NodeStatus, NodeStatus> replacements = new HashMap<NodeStatus, NodeStatus>();
            HashSet<NodeStatus> toRemove = new HashSet<NodeStatus>();
            for (NodeStatus node2 : this.nodes) {
                NodeId id = node2.getNodeId();
                if (this.nodeHealthCount.getOrDefault(id, 0) > 4) {
                    LOG.info(String.format("Removing Node %s, unhealthy threshold has been reached", node2.getExternalUri()));
                    toRemove.add(node2);
                    break;
                }
                Instant now = Instant.now();
                Instant lastTouched = this.nodePurgeTimes.getOrDefault(id, Instant.now());
                Instant lostTime = lastTouched.plus(node2.getHeartbeatPeriod().multipliedBy(2L));
                Instant deadTime = lastTouched.plus(node2.getHeartbeatPeriod().multipliedBy(4L));
                if (node2.getAvailability() == Availability.UP && lostTime.isBefore(now)) {
                    LOG.info(String.format("Switching Node %s from UP to DOWN", node2.getExternalUri()));
                    replacements.put(node2, this.rewrite(node2, Availability.DOWN));
                    this.nodePurgeTimes.put(id, Instant.now());
                    continue;
                }
                if (node2.getAvailability() != Availability.DOWN || !deadTime.isBefore(now)) continue;
                LOG.info(String.format("Removing Node %s, DOWN for too long", node2.getExternalUri()));
                toRemove.add(node2);
            }
            replacements.forEach((before, after) -> {
                this.nodes.remove(before);
                this.nodes.add((NodeStatus)after);
            });
            toRemove.forEach(node -> {
                this.nodes.remove(node);
                this.nodePurgeTimes.remove(node.getNodeId());
                this.nodeHealthCount.remove(node.getNodeId());
                this.events.fire(new NodeRemovedEvent((NodeStatus)node));
            });
        }
        finally {
            writeLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setAvailability(NodeId id, Availability availability) {
        Require.nonNull("Node ID", id);
        Require.nonNull("Availability", availability);
        Lock writeLock = this.lock.writeLock();
        writeLock.lock();
        try {
            NodeStatus node = this.getNode(id);
            if (node == null) {
                return;
            }
            if (availability.equals((Object)node.getAvailability())) {
                if (node.getAvailability() == Availability.UP) {
                    this.nodePurgeTimes.put(node.getNodeId(), Instant.now());
                }
            } else {
                LOG.info(String.format("Switching Node %s (uri: %s) from %s to %s", new Object[]{id, node.getExternalUri(), node.getAvailability(), availability}));
                NodeStatus refreshed = this.rewrite(node, availability);
                this.nodes.remove(node);
                this.nodes.add(refreshed);
                this.nodePurgeTimes.put(node.getNodeId(), Instant.now());
            }
        }
        finally {
            writeLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean reserve(SlotId slotId) {
        Lock writeLock = this.lock.writeLock();
        writeLock.lock();
        try {
            NodeStatus node = this.getNode(slotId.getOwningNodeId());
            if (node == null) {
                LOG.warning(String.format("Asked to reserve slot on node %s, but unable to find node", slotId.getOwningNodeId()));
                boolean bl = false;
                return bl;
            }
            if (!Availability.UP.equals((Object)node.getAvailability())) {
                LOG.warning(String.format("Asked to reserve a slot on node %s, but node is %s", new Object[]{slotId.getOwningNodeId(), node.getAvailability()}));
                boolean bl = false;
                return bl;
            }
            Optional<Slot> maybeSlot = node.getSlots().stream().filter(slot -> slotId.equals(slot.getId())).findFirst();
            if (!maybeSlot.isPresent()) {
                LOG.warning(String.format("Asked to reserve slot on node %s, but no slot with id %s found", node.getNodeId(), slotId));
                boolean bl = false;
                return bl;
            }
            this.reserve(node, maybeSlot.get());
            boolean bl = true;
            return bl;
        }
        finally {
            writeLock.unlock();
        }
    }

    public Set<NodeStatus> getSnapshot() {
        Lock readLock = this.lock.readLock();
        readLock.lock();
        try {
            ImmutableSet<NodeStatus> immutableSet = ImmutableSet.copyOf(this.nodes);
            return immutableSet;
        }
        finally {
            readLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private NodeStatus getNode(NodeId id) {
        Require.nonNull("Node ID", id);
        Lock readLock = this.lock.readLock();
        readLock.lock();
        try {
            NodeStatus nodeStatus = this.nodes.stream().filter(n -> n.getNodeId().equals(id)).findFirst().orElse(null);
            return nodeStatus;
        }
        finally {
            readLock.unlock();
        }
    }

    private NodeStatus rewrite(NodeStatus status, Availability availability) {
        return new NodeStatus(status.getNodeId(), status.getExternalUri(), status.getMaxSessionCount(), status.getSlots(), availability, status.getHeartbeatPeriod(), status.getSessionTimeout(), status.getVersion(), status.getOsInfo());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void release(SessionId id) {
        if (id == null) {
            return;
        }
        LOG.info("Releasing slot for session id " + String.valueOf(id));
        Lock writeLock = this.lock.writeLock();
        writeLock.lock();
        try {
            for (NodeStatus node : this.nodes) {
                for (Slot slot : node.getSlots()) {
                    if (slot.getSession() == null) continue;
                    if (!id.equals(slot.getSession().getId())) continue;
                    Slot released = new Slot(slot.getId(), slot.getStereotype(), slot.getLastStarted(), null);
                    this.amend(node.getAvailability(), node, released);
                    return;
                }
            }
        }
        finally {
            writeLock.unlock();
        }
    }

    public void reserve(NodeStatus status, Slot slot) {
        Instant now = Instant.now();
        Slot reserved = new Slot(slot.getId(), slot.getStereotype(), now, new Session(RESERVED, status.getExternalUri(), slot.getStereotype(), slot.getStereotype(), now));
        this.amend(Availability.UP, status, reserved);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setSession(SlotId slotId, Session session) {
        Require.nonNull("Slot ID", slotId);
        Lock writeLock = this.lock.writeLock();
        writeLock.lock();
        try {
            NodeStatus node = this.getNode(slotId.getOwningNodeId());
            if (node == null) {
                LOG.warning("Grid model and reality have diverged. Unable to find node " + String.valueOf(slotId.getOwningNodeId()));
                return;
            }
            Optional<Slot> maybeSlot = node.getSlots().stream().filter(slot -> slotId.equals(slot.getId())).findFirst();
            if (!maybeSlot.isPresent()) {
                LOG.warning("Grid model and reality have diverged. Unable to find slot " + String.valueOf(slotId));
                return;
            }
            Slot slot2 = maybeSlot.get();
            Session maybeSession = slot2.getSession();
            if (maybeSession == null) {
                LOG.warning("Grid model and reality have diverged. Slot is not reserved. " + String.valueOf(slotId));
                return;
            }
            if (!RESERVED.equals(maybeSession.getId())) {
                LOG.warning("Grid model and reality have diverged. Slot has session and is not reserved. " + String.valueOf(slotId));
                return;
            }
            Slot updated = new Slot(slot2.getId(), slot2.getStereotype(), session == null ? slot2.getLastStarted() : session.getStartTime(), session);
            this.amend(node.getAvailability(), node, updated);
        }
        finally {
            writeLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void updateHealthCheckCount(NodeId id, Availability availability) {
        Require.nonNull("Node ID", id);
        Require.nonNull("Availability", availability);
        Lock writeLock = this.lock.writeLock();
        writeLock.lock();
        try {
            int unhealthyCount = this.nodeHealthCount.getOrDefault(id, 0);
            if (availability.equals((Object)Availability.DOWN)) {
                this.nodeHealthCount.put(id, unhealthyCount + 1);
            }
            if (unhealthyCount <= 4 && availability.equals((Object)Availability.UP)) {
                this.nodeHealthCount.put(id, 0);
            }
        }
        finally {
            writeLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void amend(Availability availability, NodeStatus status, Slot slot) {
        HashSet<Slot> newSlots = new HashSet<Slot>(status.getSlots());
        newSlots.removeIf(s2 -> s2.getId().equals(slot.getId()));
        newSlots.add(slot);
        NodeStatus node = this.getNode(status.getNodeId());
        Lock writeLock = this.lock.writeLock();
        writeLock.lock();
        try {
            this.nodes.remove(node);
            this.nodes.add(new NodeStatus(status.getNodeId(), status.getExternalUri(), status.getMaxSessionCount(), newSlots, availability, status.getHeartbeatPeriod(), status.getSessionTimeout(), status.getVersion(), status.getOsInfo()));
        }
        finally {
            writeLock.unlock();
        }
    }
}

